# src/analysis/analysis_auto.py

from pathlib import Path
from typing import List

import pandas as pd
import matplotlib.pyplot as plt


from ..scraping.utils import get_project_root


def load_data() -> pd.DataFrame:

    root: Path = get_project_root()
    csv_path = root / "data" / "results" / "headlines_with_frames.csv"
    print(f"[AutoAnalyse] Loading data from: {csv_path}")

    df = pd.read_csv(csv_path)
    print(f"[AutoAnalyse] Loaded {len(df)} rows.")
    return df


def analyse_counts(df: pd.DataFrame) -> pd.DataFrame:

    counts = (
        df.groupby(["source", "frame"])
        .size()
        .reset_index(name="count")
        .sort_values(["source", "frame"])
    )
    print("[AutoAnalyse] Grouped counts:\n", counts)
    return counts


def plot_frame_distribution(counts_df: pd.DataFrame) -> None:

    root: Path = get_project_root()
    out_dir = root / "data" / "results" / "plots"
    out_dir.mkdir(exist_ok=True, parents=True)

    sources: List[str] = list(counts_df["source"].unique())
    print(f"[AutoAnalyse] Will plot sources: {sources}")
    for src in sources:
        sub = counts_df[counts_df["source"] == src]

        plt.figure(figsize=(7, 5))
        plt.bar(sub["frame"], sub["count"])
        plt.xticks(rotation=45, ha="right")
        plt.title(f"Framing distribution — {src}")
        plt.ylabel("Count")
        plt.tight_layout()

        outpath = out_dir / f"{src}_framing_bar.png"
        plt.savefig(outpath)
        plt.close()

        print(f"[AutoAnalyse] Saved plot for {src} -> {outpath}")


def generate_markdown_report(counts_df: pd.DataFrame) -> None:

    root: Path = get_project_root()
    out = root / "data" / "results" / "analysis_report.md"

    lines: List[str] = []
    lines.append("# UK Budget 2025 — Framing Analysis Report\n")

    for src in counts_df["source"].unique():
        lines.append(f"## Source: {src}\n")

        sub = counts_df[counts_df["source"] == src]
        total = int(sub["count"].sum())
        lines.append(f"- Total headlines: **{total}**\n")

        for _, row in sub.iterrows():
            frame = row["frame"]
            count = int(row["count"])
            lines.append(f"  - **{frame}**: {count}")
        lines.append("")

        lines.append(f"![Framing distribution](/data/results/plots/{src}_framing_bar.png)\n")

    out.write_text("\n".join(lines), encoding="utf-8")
    print(f"[AutoAnalyse] Saved markdown report -> {out}")


def main() -> None:
    df = load_data()
    counts = analyse_counts(df)
    plot_frame_distribution(counts)
    generate_markdown_report(counts)
    print("[AutoAnalyse] Done. Check data/results/plots/ and analysis_report.md")


if __name__ == "__main__":
    main()
